# PowerShell script to create XenApp Published Applications based on start menu links
# Script: StartMenuPublishing.ps1
# Author: Andreas Mariotti
# Version: 1.0, 10/18/2012


function StartMenuPublishing
{
	# Load XenApp Snap-in's
	if (-not ((Get-PSSnapin) -match "Citrix.XenApp.Commands")){Add-PSSnapin Citrix.XenApp.Commands}
	if (-not ((Get-PSSnapin) -match "Citrix.Common.Commands")){Add-PSSnapin Citrix.Common.Commands}

	# Remove all existing published applications (for testing propose only)
	# Write-Host "Removing all existing Published Applications"
	# Get-XAApplication | Remove-XAApplication
	
	# Create default worker group (if not exists)
	if (Get-XAWorkerGroup "DefaultWorkerGroup" -ErrorAction SilentlyContinue) {Write-Host "Create DefaultWorkerGroup, skiped because already exists"}
	else {New-XAWorkerGroup -WorkerGroupName "DefaultWorkerGroup" -Description "DefaultWorkerGroup"}
	
	# Add current server to default worker group
	Write-Host "Add current server to DefaultWorkerGroup" 
	Add-XAWorkerGroupServer "DefaultWorkerGroup" -ServerNames $env:computername -ErrorAction SilentlyContinue
	
	# Access start menu items via Windows Script Host
	$obj = New-Object -ComObject WScript.Shell
	$pathUser = [System.Environment]::GetFolderPath(Programs)
	$pathCommon = $obj.SpecialFolders.Item(AllUsersPrograms)
	dir $pathUser, $pathCommon -Filter *.lnk -Recurse |
	ForEach-Object	{

		# Split full name into file and folder and build proper XenApp folder and publ app names
		$link = $obj.CreateShortcut($_.FullName)
		$LinkFile = split-path $Link.FullName -leaf -resolve
		$LinkFile = $LinkFile.Replace(".lnk","")
		$LinkFolder = split-path $Link.FullName
		$LinkFolder = $LinkFolder.Replace($pathUser, "")
		$LinkFolder = $LinkFolder.Replace($pathCommon, "")
		$LinkFolder = $LinkFolder.Replace("\", "/")
		$LinkFolder = "Applications$LinkFolder"

		# Remove unsupported characters from publ app name
		$LinkFile = $LinkFile.Replace("\","")
		$LinkFile = $LinkFile.Replace("/","")
		$LinkFile = $LinkFile.Replace(";","")
		$LinkFile = $LinkFile.Replace(":","")
		$LinkFile = $LinkFile.Replace("#","")
		$LinkFile = $LinkFile.Replace(".","")
		$LinkFile = $LinkFile.Replace("*","")
		$LinkFile = $LinkFile.Replace("?","")
		$LinkFile = $LinkFile.Replace("<","")
		$LinkFile = $LinkFile.Replace(">","")
		$LinkFile = $LinkFile.Replace("|","")
		$LinkFile = $LinkFile.Replace("[","")
		$LinkFile = $LinkFile.Replace("]","")
		$LinkFile = $LinkFile.Replace("(","")
		$LinkFile = $LinkFile.Replace(")","")
		$LinkFile = $LinkFile.Replace("""","")
		$LinkFile = $LinkFile.Replace("'","")
		
		# Add arguments to executable (if exists)
		$Executable = $link.TargetPath
		if ($Link.Arguments -ne "") {$Executable = $link.TargetPath + " " + $link.Arguments}

		# Fallback do default icon for links without proper icons (i.e. direct links to .msc files)
		$LinkIcon = $Link.IconLocation.split(",")
		$IconFile = $LinkIcon[0]
		$IconIndex = $LinkIcon[1]
		if ($LinkIcon[0] -eq "") {$IconFile = $link.TargetPath}		
		if ($LinkIcon[1] -eq "") {$IconIndex = 0}		
		if ([int]$LinkIcon[1] -lt 0) {$IconIndex = 0}
		$IconExt = [System.IO.Path]::GetExtension($IconFile)
		if ($IconExt -contains ".msc") {$IconFile = "mmc.exe"}
		
		# Create XenApp Published Application
		if ($link.TargetPath -ne "") 
		{
			Write-Host "Create" $LinkFolder"/"$LinkFile
			New-XAApplication -Force -SkipPassThru -ApplicationType ServerInstalled -DisplayName $LinkFile -CommandLineExecutable $Executable -WorkingDirectory $link.WorkingDirectory -WorkerGroupNames DefaultWorkerGroup -Description $Link.Description -FolderPath $LinkFolder -ClientFolder $LinkFolder.Replace("/", "\") -StartMenuFolder $LinkFolder.Replace("/", "\") -Accounts "$env:logondomain\Domain Users" -EncodedIconData (Get-CtxIcon $IconFile -index $IconIndex) -ErrorAction SilentlyContinue
		}		
		
	}
}
 
StartMenuPublishing